using System.Collections.Generic;
using UnityEngine;
using UnityEngine.XR.ARFoundation;

namespace Modules.MachineRecognition.Scripts
{
    /// <summary>
    /// Manages the tracking of images in an AR environment.
    /// Handles adding, updating, and removing tracked images, as well as initializing their controllers.
    /// </summary>
    public class ImageTrackingManager : MonoBehaviour
    {
        /// <summary>
        /// The ARTrackedImageManager responsible for tracking images in the AR environment.
        /// </summary>
        [SerializeField] private ARTrackedImageManager trackedImageManager;

        /// <summary>
        /// The prefab to instantiate for each tracked image.
        /// </summary>
        [SerializeField] private GameObject trackedImagePrefab;

        /// <summary>
        /// The audio clip to play when a new tracked image is added.
        /// </summary>
        [SerializeField] private AudioClip trackedImageAlert;

        /// <summary>
        /// The AudioSource used to play the alert sound.
        /// </summary>
        private AudioSource _audioSource;

        /// <summary>
        /// A dictionary mapping image names to their corresponding TrackedImageController instances.
        /// </summary>
        private Dictionary<string, TrackedImageController> _trackedImages = new();

        /// <summary>
        /// Subscribes to the trackedImagesChanged event when the object is enabled.
        /// </summary>
        private void OnEnable() => trackedImageManager.trackedImagesChanged += OnTrackedImagesChanged;

        /// <summary>
        /// Unsubscribes from the trackedImagesChanged event when the object is disabled.
        /// </summary>
        private void OnDisable() => trackedImageManager.trackedImagesChanged -= OnTrackedImagesChanged;

        /// <summary>
        /// Initializes the AudioSource component and sets its clip to the tracked image alert sound.
        /// </summary>
        private void Awake()
        {
            _audioSource = gameObject.AddComponent<AudioSource>();
            _audioSource.clip = trackedImageAlert;
        }

        /// <summary>
        /// Handles changes to the tracked images, including added, updated, and removed images.
        /// </summary>
        /// <param name="args">The event arguments containing the added, updated, and removed tracked images.</param>
        private void OnTrackedImagesChanged(ARTrackedImagesChangedEventArgs args)
        {
            foreach (var trackedImage in args.added)
            {
                #if UNITY_ANDROID && !UNITY_EDITOR
                            Handheld.Vibrate();
                #endif

                UpdateImage(trackedImage);
            }

            foreach (var trackedImage in args.updated)
                UpdateImage(trackedImage);

            foreach (var trackedImage in args.removed)
                RemoveImage(trackedImage);
        }

        /// <summary>
        /// Updates or initializes the controller for a tracked image.
        /// If the image is not already tracked, a new controller is created and initialized.
        /// </summary>
        /// <param name="trackedImage">The ARTrackedImage to update or initialize.</param>
        private void UpdateImage(ARTrackedImage trackedImage)
        {
            var imageName = trackedImage.referenceImage.name;

            if (!_trackedImages.TryGetValue(imageName, out var controller))
            {
                var rotation = Quaternion.Euler(0, trackedImage.transform.rotation.eulerAngles.y - 180f, 0);
                var newObj = Instantiate(trackedImagePrefab, trackedImage.transform.position, rotation);

                controller = newObj.GetComponent<TrackedImageController>();
                controller.Initialize(trackedImage);

                _audioSource?.Play();
                _trackedImages[imageName] = controller;
            }

            controller.SetTrackedImage(trackedImage);
        }

        /// <summary>
        /// Removes a tracked image and hides its associated controller.
        /// </summary>
        /// <param name="trackedImage">The ARTrackedImage to remove.</param>
        private void RemoveImage(ARTrackedImage trackedImage)
        {
            var imageName = trackedImage.referenceImage.name;

            if (_trackedImages.TryGetValue(imageName, out var controller))
            {
                controller.Hide();
            }
        }
    }
}